#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <QmiService.h>
#include <CtlService.h>
#include <QmiSyncObject.h>
#include "lite-qmi-dms.h"
#include "dms.h"
#include "fms.h"

typedef int (* pack_func) (pack_qmi_t*, uint8_t*, uint16_t*, void*);
typedef int (* unpack_func) (uint8_t*, uint16_t, void*);

#define pack_func_item pack_func* 
#define unpack_func_item unpack_func*
#define run_pack_item(x) ((pack_func) x)
#define run_unpack_item(x) ((unpack_func) x)

typedef struct
{
    pack_func_item   pack;
    char pack_func_name[128];
    void*       pack_ptr;
    unpack_func_item unpack;
    char unpack_func_name[128];
    void*       unpack_ptr;
} testitem_t;

enum eQCWWANError ExecuteDmsTestCase(QmiService* pQmiService, testitem_t *pTestParams);

extern void DisplayGetBandCapability(unpack_dms_SLQSGetBandCapability_t *pBandCapability);
extern void DisplayGetFirmwareInformation(unpack_dms_GetFirmwareInfo_t *pfirmwareInfo);
extern void DisplayGetSerialNumbers(unpack_dms_GetSerialNumbers_t *pSerialNumbers);
extern void DisplayGetHostDevInfoSettings(unpack_dms_SLQSSwiGetHostDevInfo_t *pSwiGetHostDevInfo);
extern void DisplaySetEventReportInd(unpack_dms_SetEventReport_ind_t *pSetEventReport_ind);
extern void DisplayGetPower(unpack_dms_GetPower_t *pUnpackGetPower);
void DeleteStoredImage(QmiService* pQmiService, pack_fms_DeleteStoredImage_t * pDeleteStoredImage)
{
	testitem_t testItem;
	unpack_fms_DeleteStoredImage_t unpackfmsDeleteStoredImage;
	memset(&unpackfmsDeleteStoredImage, 0, sizeof(unpack_fms_DeleteStoredImage_t));

	testItem.pack = (pack_func_item)&pack_fms_DeleteStoredImage;
	strcpy(testItem.pack_func_name, "pack_fms_DeleteStoredImage");
	testItem.pack_ptr = pDeleteStoredImage;
	testItem.unpack = (unpack_func_item)&unpack_fms_DeleteStoredImage;
	strcpy(testItem.unpack_func_name, "unpack_dms_SLQSSwiSetHunpack_fms_DeleteStoredImageostDevInfo");
	testItem.unpack_ptr = &unpackfmsDeleteStoredImage;

	if (ExecuteDmsTestCase(pQmiService, &testItem) == eQCWWAN_ERR_NONE)
	{
		printf(ANSI_COLOR_GREEN);
		swi_uint256_print_mask(unpackfmsDeleteStoredImage.ParamPresenceMask);
		printf("%s Result: %d\n", __FUNCTION__, unpackfmsDeleteStoredImage.Tlvresult);
		printf(ANSI_COLOR_RESET);
	}
}

void GetStoredImages(QmiService* pQmiService)
{
    testitem_t testItem;
    unpack_fms_GetStoredImages_t stored_images;
    memset(&stored_images, 0, sizeof(unpack_fms_GetStoredImages_t));

    testItem.pack = (pack_func_item) &pack_fms_GetStoredImages;
    strcpy(testItem.pack_func_name, "pack_fms_GetStoredImages");
    testItem.pack_ptr = NULL;
    testItem.unpack = (unpack_func_item) &unpack_fms_GetStoredImages;
    strcpy(testItem.unpack_func_name, "unpack_fms_GetStoredImages");
    testItem.unpack_ptr = &stored_images;

    if (ExecuteDmsTestCase(pQmiService, &testItem) == eQCWWAN_ERR_NONE)
    {
        printf("%s: %s\n\n",__FUNCTION__, stored_images.Tlvresult? "Fail" : "Success");
        if(swi_uint256_get_bit (stored_images.ParamPresenceMask, 1))
        {
            for (uint32_t i = 0; i < stored_images.imageList.listSize; i++)
            {
                for (uint8_t j = 0; j < stored_images.imageList.imageIDEntries[i].imageIDSize; j++)
                {
                    printf("  Type: %s\n", stored_images.imageList.imageIDEntries[i].imageType == 0 ? "Modem image" : "PRI image");
                    printf("  Image ID: %s\n", stored_images.imageList.imageIDEntries[i].imageIDElement[j].imageID);
                    printf("  Bulid ID: %s\n\n", stored_images.imageList.imageIDEntries[i].imageIDElement[j].buildID);
                }
            }
        }
    }
}

void GetModelID(QmiService* pQmiService)
{
    testitem_t testItem;
    unpack_dms_GetModelID_t model;
    memset(&model, 0, sizeof(unpack_dms_GetModelID_t));

    testItem.pack = (pack_func_item) &pack_dms_GetModelID;
    strcpy(testItem.pack_func_name, "pack_dms_GetModelID");
    testItem.pack_ptr = NULL;
    testItem.unpack = (unpack_func_item) &unpack_dms_GetModelID;
    strcpy(testItem.unpack_func_name, "unpack_dms_GetModelID");
    testItem.unpack_ptr = &model;

    if (ExecuteDmsTestCase(pQmiService, &testItem) == eQCWWAN_ERR_NONE)
    {
        printf("%s: %s\n",__FUNCTION__, model.Tlvresult? "Fail" : "Success");
        if(swi_uint256_get_bit (model.ParamPresenceMask, 1))
            printf(" Model ID: %s\n", model.modelid);
    }
}

void GetFsn(QmiService* pQmiService)
{
    testitem_t testItem;
    unpack_dms_GetFSN_t FSN;
    memset(&FSN, 0, sizeof(unpack_dms_GetFSN_t));

    testItem.pack = (pack_func_item) &pack_dms_GetFSN;
    strcpy(testItem.pack_func_name, "pack_dms_GetFSN");
    testItem.pack_ptr = NULL;
    testItem.unpack = (unpack_func_item) &unpack_dms_GetFSN;
    strcpy(testItem.unpack_func_name, "unpack_dms_GetFSN");
    testItem.unpack_ptr = &FSN;

    if (ExecuteDmsTestCase(pQmiService, &testItem) == eQCWWAN_ERR_NONE)
    {
        printf("%s: %s\n",__FUNCTION__, FSN.Tlvresult? "Fail" : "Success");
        if(swi_uint256_get_bit (FSN.ParamPresenceMask, 1))
            printf(" FSN: %s\n", FSN.String);
    }
}

void GetDeviceCapabilities(QmiService* pQmiService)
{
    testitem_t testItem;
    unpack_dms_GetDeviceCapabilities_t deviceCap;
    memset(&deviceCap, 0, sizeof(unpack_dms_GetDeviceCapabilities_t));

    testItem.pack = (pack_func_item) &pack_dms_GetDeviceCapabilities;
    strcpy(testItem.pack_func_name, "pack_dms_GetDeviceCapabilities");
    testItem.pack_ptr = NULL;
    testItem.unpack = (unpack_func_item) &unpack_dms_GetDeviceCapabilities;
    strcpy(testItem.unpack_func_name, "unpack_dms_GetDeviceCapabilities");
    testItem.unpack_ptr = &deviceCap;

    if (ExecuteDmsTestCase(pQmiService, &testItem) == eQCWWAN_ERR_NONE)
    {
        uint8_t idx = 0;
        printf("\n%s: Success \n",__FUNCTION__);
        printf(" Max TX Channel Rate : %X\n", deviceCap.maxTxChannelRate );
        printf(" Max RX Channel Rate : %X\n", deviceCap.maxRxChannelRate );
        printf(" Data Service Capability : %X\n", deviceCap.dataServiceCaCapability );
        printf(" SIM Capability : %X\n", deviceCap.simCapability );
        printf(" Radio Interfaces Size: %X\n", deviceCap.radioIfacesSize);
        for(idx = 0; idx < deviceCap.radioIfacesSize; idx++)
        {
            printf("  radio Interface[%d] : %X\n", idx, deviceCap.RadioIfaces[idx]);
        }
    }
}

void GetBandCapability(QmiService* pQmiService)
{
    testitem_t testItem;
    
    uint16_t lteBands[32];
    uint16_t nr5gBands[32];
    unpack_dms_SLQSGetBandCapability_t dmsGetBandCapability = {
                                        0, 0, 0, 0, 0, 
                                        0, { 32, lteBands },
                                        0, { 32, nr5gBands },
                                        SWI_UINT256_INT_VALUE};

    testItem.pack = (pack_func_item) &pack_dms_SLQSGetBandCapability;
    strcpy(testItem.pack_func_name, "pack_dms_SLQSGetBandCapability");
    testItem.pack_ptr = NULL;
    testItem.unpack = (unpack_func_item) &unpack_dms_SLQSGetBandCapability;
    strcpy(testItem.unpack_func_name, "unpack_dms_SLQSGetBandCapability");
    testItem.unpack_ptr = &dmsGetBandCapability;

    if (ExecuteDmsTestCase(pQmiService, &testItem) == eQCWWAN_ERR_NONE)
    {
        DisplayGetBandCapability(&dmsGetBandCapability);
    }
}

void GetManufacturer(QmiService* pQmiService)
{
    testitem_t testItem;
    unpack_dms_GetManufacturer_t    getManufacturer;
    memset(&getManufacturer, 0, sizeof(unpack_dms_GetManufacturer_t));

    testItem.pack = (pack_func_item) &pack_dms_GetManufacturer;
    strcpy(testItem.pack_func_name, "pack_dms_GetManufacturer");
    testItem.pack_ptr = NULL;
    testItem.unpack = (unpack_func_item) &unpack_dms_GetManufacturer;
    strcpy(testItem.unpack_func_name, "pack_dms_GetManufacturer");
    testItem.unpack_ptr = &getManufacturer;

    if (ExecuteDmsTestCase(pQmiService, &testItem) == eQCWWAN_ERR_NONE)
    {
        printf(" %s: %s\n",__FUNCTION__, getManufacturer.Tlvresult? "Fail" : "Success");
        if((getManufacturer.Tlvresult == eQCWWAN_ERR_NONE) && (swi_uint256_get_bit (getManufacturer.ParamPresenceMask, 1)))
            printf(" Device Manufacturer: %s\n", getManufacturer.manufacturer);
    }
}

void GetMsisdn(QmiService* pQmiService)
{
    testitem_t testItem;
    unpack_dms_GetVoiceNumber_t voiceNumber;
    memset(&voiceNumber, 0, sizeof(unpack_dms_GetVoiceNumber_t));

    testItem.pack = (pack_func_item) &pack_dms_GetVoiceNumber;
    strcpy(testItem.pack_func_name, "pack_dms_GetVoiceNumber");
    testItem.pack_ptr = NULL;
    testItem.unpack = (unpack_func_item) &unpack_dms_GetVoiceNumber;
    strcpy(testItem.unpack_func_name, "unpack_dms_GetVoiceNumber");
    testItem.unpack_ptr = &voiceNumber;

    if (ExecuteDmsTestCase(pQmiService, &testItem) == eQCWWAN_ERR_NONE)
    {
        printf (ANSI_COLOR_GREEN);
        printf("%s: %s\n",__FUNCTION__, voiceNumber.Tlvresult? "Fail" : "Success");
        if(swi_uint256_get_bit (voiceNumber.ParamPresenceMask, 16))
            printf("\tMobile ID: %s\n", voiceNumber.MIN);
        if(swi_uint256_get_bit (voiceNumber.ParamPresenceMask, 1))
            printf("\tVoice Number: %s\n", voiceNumber.VoiceNumber);
        printf (ANSI_COLOR_RESET);
    }
}

void GetFirmwareInformation(QmiService* pQmiService)
{
    testitem_t testItem;
    unpack_dms_GetFirmwareInfo_t firmwareInfo;
    memset(&firmwareInfo, 0, sizeof(unpack_dms_GetFirmwareInfo_t));

    testItem.pack = (pack_func_item) &pack_dms_SLQSSwiGetFirmwareInfo;
    strcpy(testItem.pack_func_name, "pack_dms_SLQSSwiGetFirmwareInfo");
    testItem.pack_ptr = NULL;
    testItem.unpack = (unpack_func_item) &unpack_dms_SLQSSwiGetFirmwareInfo;
    strcpy(testItem.unpack_func_name, "unpack_dms_SLQSSwiGetFirmwareInfo");
    testItem.unpack_ptr = &firmwareInfo;

    if (ExecuteDmsTestCase(pQmiService, &testItem) == eQCWWAN_ERR_NONE)
    {
        printf (ANSI_COLOR_GREEN);
        DisplayGetFirmwareInformation(&firmwareInfo);
        printf (ANSI_COLOR_RESET);
    }
}

void GetSerialNumber(QmiService* pQmiService)
{
    testitem_t testItem;
    unpack_dms_GetSerialNumbers_t serialNumbers;
    memset(&serialNumbers, 0, sizeof(unpack_dms_GetSerialNumbers_t));

    testItem.pack = (pack_func_item) &pack_dms_GetSerialNumbers;
    strcpy(testItem.pack_func_name, "pack_dms_GetSerialNumbers");
    testItem.pack_ptr = NULL;
    testItem.unpack = (unpack_func_item) &unpack_dms_GetSerialNumbers;
    strcpy(testItem.unpack_func_name, "unpack_dms_GetSerialNumbers");
    testItem.unpack_ptr = &serialNumbers;
    
    if (ExecuteDmsTestCase(pQmiService, &testItem) == eQCWWAN_ERR_NONE)
    {
        printf (ANSI_COLOR_GREEN);
        DisplayGetSerialNumbers(&serialNumbers);
        printf (ANSI_COLOR_RESET);
    }
}

void GetSwiHostDevInfo(QmiService* pQmiService)
{
	testitem_t testItem;
	pack_dms_SLQSSwiGetHostDevInfo_t packdmsSLQSSwiGetHostDevInfo = { 0 };
	unpack_dms_SLQSSwiGetHostDevInfo_t unpackdmsSLQSSwiGetHostDevInfo;
	memset(&unpackdmsSLQSSwiGetHostDevInfo, 0, sizeof(unpack_dms_SLQSSwiGetHostDevInfo_t));

	testItem.pack = (pack_func_item)&pack_dms_SLQSSwiGetHostDevInfo;
	strcpy(testItem.pack_func_name, "pack_dms_SLQSSwiGetHostDevInfo");
	testItem.pack_ptr = &packdmsSLQSSwiGetHostDevInfo;
	testItem.unpack = (unpack_func_item)&unpack_dms_SLQSSwiGetHostDevInfo;
	strcpy(testItem.unpack_func_name, "unpack_dms_SLQSSwiGetHostDevInfo");
	testItem.unpack_ptr = &unpackdmsSLQSSwiGetHostDevInfo;

	if (ExecuteDmsTestCase(pQmiService, &testItem) == eQCWWAN_ERR_NONE)
	{
		printf(ANSI_COLOR_GREEN);
		DisplayGetHostDevInfoSettings(&unpackdmsSLQSSwiGetHostDevInfo);
		printf(ANSI_COLOR_RESET);
	}
}

void SetSwiHostDevInfo(QmiService* pQmiService, pack_dms_SLQSSwiSetHostDevInfo_t *pSwiSetHostDevInfo)
{
	testitem_t testItem;
	unpack_dms_SLQSSwiSetHostDevInfo_t unpackdmsSLQSSwiSetHostDevInfo;
	memset(&unpackdmsSLQSSwiSetHostDevInfo, 0, sizeof(unpack_dms_SLQSSwiSetHostDevInfo_t));

	testItem.pack = (pack_func_item)&pack_dms_SLQSSwiSetHostDevInfo;
	strcpy(testItem.pack_func_name, "pack_dms_SLQSSwiSetHostDevInfo");
	testItem.pack_ptr = pSwiSetHostDevInfo;
	testItem.unpack = (unpack_func_item)&unpack_dms_SLQSSwiSetHostDevInfo;
	strcpy(testItem.unpack_func_name, "unpack_dms_SLQSSwiSetHostDevInfo");
	testItem.unpack_ptr = &unpackdmsSLQSSwiSetHostDevInfo;

	if (ExecuteDmsTestCase(pQmiService, &testItem) == eQCWWAN_ERR_NONE)
	{
		printf(ANSI_COLOR_GREEN);
		swi_uint256_print_mask(unpackdmsSLQSSwiSetHostDevInfo.ParamPresenceMask);
		printf("%s Result: %d\n", __FUNCTION__, unpackdmsSLQSSwiSetHostDevInfo.Tlvresult);
		printf(ANSI_COLOR_RESET);
	}
}

void SetEventReport(QmiService* pQmiService, pack_dms_SetEventReport_t *pSetEventReport)
{
	testitem_t testItem;
    unpack_dms_SetEventReport_t sUnpackSetEventReport;
	memset(&sUnpackSetEventReport, 0, sizeof(unpack_dms_SetEventReport_t));

	testItem.pack = (pack_func_item)&pack_dms_SetEventReport;
	strcpy(testItem.pack_func_name, "pack_dms_SetEventReport");
	testItem.pack_ptr = pSetEventReport;
	testItem.unpack = (unpack_func_item)&unpack_dms_SetEventReport;
	strcpy(testItem.unpack_func_name, "unpack_dms_SetEventReport");
	testItem.unpack_ptr = &sUnpackSetEventReport;

	if (ExecuteDmsTestCase(pQmiService, &testItem) == eQCWWAN_ERR_NONE)
	{
		printf(ANSI_COLOR_GREEN);
		swi_uint256_print_mask(sUnpackSetEventReport.ParamPresenceMask);
		printf("%s Result: %d\n", __FUNCTION__, sUnpackSetEventReport.Tlvresult);
		printf(ANSI_COLOR_RESET);
	}
}

void SetPower(QmiService* pQmiService, pack_dms_SetPower_t *pSetPower)
{
	testitem_t testItem;
    unpack_dms_SetPower_t sUnpackSetPower;
	memset(&sUnpackSetPower, 0, sizeof(unpack_dms_SetPower_t));

	testItem.pack = (pack_func_item)&pack_dms_SetPower;
	strcpy(testItem.pack_func_name, "pack_dms_SetPower");
	testItem.pack_ptr = pSetPower;
	testItem.unpack = (unpack_func_item)&unpack_dms_SetPower;
	strcpy(testItem.unpack_func_name, "unpack_dms_SetPower");
	testItem.unpack_ptr = &sUnpackSetPower;

	if (ExecuteDmsTestCase(pQmiService, &testItem) == eQCWWAN_ERR_NONE)
	{
		printf(ANSI_COLOR_GREEN);
		swi_uint256_print_mask(sUnpackSetPower.ParamPresenceMask);
		printf("%s Result: %d\n", __FUNCTION__, sUnpackSetPower.Tlvresult);
		printf(ANSI_COLOR_RESET);
	}
}

void GetPower(QmiService* pQmiService)
{
	testitem_t testItem;
    unpack_dms_GetPower_t sUnpackGetPower;
	memset(&sUnpackGetPower, 0, sizeof(unpack_dms_GetPower_t));

	testItem.pack = (pack_func_item)&pack_dms_GetPower;
	strcpy(testItem.pack_func_name, "pack_dms_GetPower");
	testItem.pack_ptr = NULL;
	testItem.unpack = (unpack_func_item)&unpack_dms_GetPower;
	strcpy(testItem.unpack_func_name, "unpack_dms_GetPower");
	testItem.unpack_ptr = &sUnpackGetPower;

	if (ExecuteDmsTestCase(pQmiService, &testItem) == eQCWWAN_ERR_NONE)
	{
		printf(ANSI_COLOR_GREEN);
		DisplayGetPower(&sUnpackGetPower);
		printf(ANSI_COLOR_RESET);
	}
}

void dms_indication_handler(uint16_t msgid, uint8_t *qmiData, uint32_t qmiDataSize)
{
    int rtn = 0;

    switch (msgid)
    {
        case eQMI_DMS_EVENT_IND:
        {
            unpack_dms_SetEventReport_ind_t sSetEventReport_ind;
            memset(&sSetEventReport_ind, 0, sizeof(unpack_dms_SetEventReport_ind_t));
            rtn = unpack_dms_SetEventReport_ind(qmiData, qmiDataSize, &sSetEventReport_ind);
            if (rtn != eQCWWAN_ERR_NONE)
            {
                printf (ANSI_COLOR_RED);
                printf("\nunpack_dms_SetEventReport_ind errorCode: %d\n", rtn);
            }
            else
            {
                DisplaySetEventReportInd(&sSetEventReport_ind);
            }
        }
        break;
        case eQMI_DMS_GET_MODEL_ID:
        case eQMI_DMS_GET_REV_ID:
        case eQMI_DMS_PSM_CFG_PARAMS_CHANGE_IND:
        case eQMI_DMS_SWI_EVENT_IND:
        case eQMI_DMS_SWI_GET_RESET_INFO_IND:
        break;
    }
    printf (ANSI_COLOR_RESET);
}

enum eQCWWANError ExecuteDmsTestCase(QmiService* pQmiService, testitem_t *pTestParams)
{
    enum eQCWWANError qcError = eQCWWAN_ERR_NONE;
    int ret = FAILURE;
    pack_qmi_t  req_ctx;
    
    QmiSyncObject syncObject;
    QmiSyncObject_Initialize(&syncObject);

    memset(&req_ctx, 0, sizeof(req_ctx));	

    // Obtain transaction ID.
    req_ctx.xid = QmiService_GetNextTransactionId(pQmiService);
	
    qcError = run_pack_item(pTestParams->pack)(&req_ctx, syncObject.buffer, &syncObject.bufferSize, pTestParams->pack_ptr);
    if(qcError != eQCWWAN_ERR_NONE)
 	{	
        printf (ANSI_COLOR_RED);
        printf ("packing function %s returned error %d: %s\n", 
                    pTestParams->pack_func_name,
                    qcError, helper_get_error_reason(qcError));
        QmiSyncObject_Destroy(&syncObject);
        printf (ANSI_COLOR_RESET);
        return qcError;
	}	
	
	QmiSyncObject_Lock(&syncObject);

    ret = QmiService_SendRequest(
		pQmiService, 
		req_ctx.xid, 
		syncObject.buffer, 
		syncObject.bufferSize, 
		QmiSyncObject_ResponseCallback, 
		(void*)&syncObject
		);

	if (ret != SUCCESS)
	{
        printf (ANSI_COLOR_RED);
        printf("error, failed to run qmi command\n");
		QmiSyncObject_Unlock(&syncObject);
	}
	else
	{
		ret = QmiSyncObject_TimedWait(&syncObject, 10);			
		QmiSyncObject_Unlock(&syncObject);

		if (ret != SUCCESS)
		{
            printf (ANSI_COLOR_RED);
			printf ("Function timed out\n");
			QmiService_CancelTransaction(pQmiService, req_ctx.xid);	
		}
		else
		{
            if (pTestParams->unpack)
            {
                qcError = run_unpack_item (pTestParams->unpack) (syncObject.buffer, syncObject.bufferSize, pTestParams->unpack_ptr);	
                if (qcError != eQCWWAN_ERR_NONE)
                {	
                    printf (ANSI_COLOR_RED);
                    printf ( "unpack function %s returned error %d: %s\n", 
                        pTestParams->unpack_func_name, qcError, helper_get_error_reason(qcError));
                }
            }
            else 
            {
                #ifdef TEST_DBG
                fprintf (stderr, "no unpack function\n");
                #endif
            }
		}
	}

    QmiSyncObject_Destroy(&syncObject);
    printf (ANSI_COLOR_RESET);

    return qcError;
}
